/*
 * Decompiled with CFR 0.152.
 */
package open.batoru.core;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import open.batoru.catalog.CardSorter;
import open.batoru.data.Card;
import open.batoru.data.CardConst;
import open.batoru.data.CardDataColor;
import open.batoru.data.CardDataImageSet;
import open.batoru.data.CardLoader;
import open.batoru.parsers.RestrictedCardsParser;

public final class Deck {
    private final List<String> deckMain = new ArrayList<String>();
    private final List<String> deckLRIG = new ArrayList<String>();
    private final Map<String, String> specialRestrictMarkedMap = new HashMap<String, String>();
    private final File sourceFile;

    public Deck(File sourceFile) {
        this.sourceFile = sourceFile;
    }

    public Deck(Deck source) {
        this.sourceFile = source.getSourceFile();
        List<String> deckList = source.getDeckList();
        for (String imageSet : deckList) {
            this.addCard(imageSet);
        }
    }

    public File getSourceFile() {
        return this.sourceFile;
    }

    public String getSourceFileName() {
        return this.sourceFile != null ? this.sourceFile.getName() : "";
    }

    public int addCard(String imageSet) {
        List<String> deckArray;
        DeckType deckType;
        Card cardObject = CardLoader.getCardByImageSet(imageSet);
        if (cardObject == null || cardObject.getBanlistRestriction() == RestrictedCardsParser.RestrictType.TOKEN) {
            return -1;
        }
        if (Deck.isMainDeckCard(cardObject)) {
            deckType = DeckType.MAIN;
            deckArray = this.deckMain;
        } else {
            deckType = DeckType.LRIG;
            deckArray = this.deckLRIG;
        }
        if (deckType == DeckType.MAIN && this.getCardCount(cardObject) >= 4 || deckType == DeckType.LRIG && this.getCardCount(cardObject) >= 1) {
            return -1;
        }
        deckArray.add(imageSet);
        if (cardObject.getDoubleFacedCardPairObject() != null) {
            deckArray.add(CardDataImageSet.getDoubleFacedPairImageSet(cardObject, imageSet));
        }
        this.sortDeck(deckType);
        return deckArray.lastIndexOf(imageSet);
    }

    public void removeCard(String imageSet) {
        List<String> deck;
        if (this.deckMain.contains(imageSet)) {
            deck = this.deckMain;
        } else if (this.deckLRIG.contains(imageSet)) {
            deck = this.deckLRIG;
        } else {
            return;
        }
        deck.remove(imageSet);
        Card cardObject = CardLoader.getCardByImageSet(imageSet);
        if (cardObject.getDoubleFacedCardPairObject() != null) {
            deck.remove(deck.lastIndexOf(CardDataImageSet.getDoubleFacedPairImageSet(cardObject, imageSet)));
        }
        this.sortDeck(deck == this.deckMain ? DeckType.MAIN : DeckType.LRIG);
    }

    public void clearDeck() {
        this.deckMain.clear();
        this.deckLRIG.clear();
    }

    public void markCardAsSpecialRestricted(String imageSet, String sourceCardName, String restrictText) {
        String oldRestrictText = this.specialRestrictMarkedMap.get(imageSet);
        restrictText = "- \"" + (String)restrictText + "\"" + (String)(sourceCardName != null ? "(" + sourceCardName + ")" : "");
        if (oldRestrictText != null) {
            restrictText = oldRestrictText + "\n" + (String)restrictText;
        }
        this.specialRestrictMarkedMap.put(imageSet, (String)restrictText);
    }

    public void unmarkSpecialRestrictedCard(String imageSet) {
        this.specialRestrictMarkedMap.remove(imageSet);
    }

    public boolean isCardMarkedAsSpecialRestricted(String imageSet) {
        return this.specialRestrictMarkedMap.containsKey(imageSet);
    }

    public String getSpecialRestrictText(String imageSet) {
        return this.specialRestrictMarkedMap.get(imageSet);
    }

    public void clearSpecialRestrictedMarkedCards() {
        this.specialRestrictMarkedMap.clear();
    }

    public int getCardCount(Card cardObject) {
        List<String> deckArray = Deck.isMainDeckCard(cardObject) ? this.deckMain : this.deckLRIG;
        return (int)deckArray.stream().filter(imageSet -> CardLoader.getCardByImageSet(imageSet) == cardObject).count();
    }

    public int getTotalCards(DeckType deckType) {
        List<String> deckArray = deckType == DeckType.MAIN ? this.deckMain : this.deckLRIG;
        return (int)deckArray.stream().filter(this::isNotDoubleFacedCardPair).count();
    }

    public void sortDeck(DeckType deckType) {
        CardSorter.sortDeck(deckType == DeckType.MAIN ? this.deckMain : this.deckLRIG);
    }

    public void shuffleDeck() {
        Collections.shuffle(this.deckMain);
    }

    public List<String> getDeckList() {
        return Stream.concat(this.deckLRIG.stream(), this.deckMain.stream()).filter(this::isNotDoubleFacedCardPair).toList();
    }

    public List<String> getDeckList(DeckType deckType) {
        return (deckType == DeckType.MAIN ? this.deckMain : this.deckLRIG).stream().filter(this::isNotDoubleFacedCardPair).toList();
    }

    public int getLifeBurstCount() {
        return (int)this.deckMain.stream().filter(imageSet -> CardLoader.getCardByImageSet(imageSet).hasLifeBurst()).count();
    }

    public int getZeroLevelLRIGCount() {
        return (int)this.deckLRIG.stream().filter(imageSet -> CardLoader.getCardByImageSet(imageSet).getType() == CardConst.CardType.LRIG && CardLoader.getCardByImageSet(imageSet).getLevel() == 0).count();
    }

    public int getPieceTeamLimitCount() {
        return (int)this.deckLRIG.stream().filter(imageSet -> CardLoader.getCardByImageSet(imageSet).getType() == CardConst.CardType.PIECE && CardLoader.getCardByImageSet(imageSet).getLRIGTeam() != null).count();
    }

    public List<CardConst.CardLRIGType> checkDivaMatchingLRIGColors() {
        HashMap<CardConst.CardLRIGType, CardDataColor> data = new HashMap<CardConst.CardLRIGType, CardDataColor>();
        ArrayList<CardConst.CardLRIGType> cacheMismatchedLRIGTypes = new ArrayList<CardConst.CardLRIGType>();
        for (String imageSet : this.deckLRIG) {
            Card cardObject = CardLoader.getCardByImageSet(imageSet);
            if (!CardConst.CardType.isLRIG(cardObject.getType()) || cardObject.getColor().getPrimaryValue() == CardConst.CardColor.COLORLESS || cardObject.getLRIGType().getValue().size() != 1) continue;
            CardDataColor colors = (CardDataColor)data.get((Object)cardObject.getLRIGType().getPrimaryValue());
            if (colors != null && !colors.matches(cardObject.getColor())) {
                cacheMismatchedLRIGTypes.add(cardObject.getLRIGType().getPrimaryValue());
                continue;
            }
            data.putIfAbsent(cardObject.getLRIGType().getPrimaryValue(), cardObject.getColor());
        }
        return cacheMismatchedLRIGTypes;
    }

    public boolean containsSpecialRestrictedCards() {
        this.specialRestrictMarkedMap.clear();
        boolean found = false;
        for (String imageSet : this.getDeckList()) {
            Card cardObject = CardLoader.getCardByImageSet(imageSet);
            if (cardObject.getSpecialRestriction() == null || !cardObject.getSpecialRestriction().check(imageSet, this)) continue;
            found = true;
        }
        return found;
    }

    public boolean containsRestrictedCards() {
        ArrayList<Card> listRestrictSimultaneous = new ArrayList<Card>();
        List<String> deckList = this.getDeckList();
        for (String imageSet2 : deckList) {
            Card cardObject = CardLoader.getCardByImageSet(imageSet2);
            if (listRestrictSimultaneous.contains(cardObject) || cardObject.getBanlistRestriction() != RestrictedCardsParser.RestrictType.NONE && cardObject.getBanlistRestriction() != RestrictedCardsParser.RestrictType.UP_TO_TWO) {
                return true;
            }
            if (cardObject.getBanlistSimultaneousNum() <= 0) continue;
            listRestrictSimultaneous.addAll(cardObject.getBanlistSimultaneousList());
        }
        return this.getDeckList().stream().anyMatch(imageSet -> listRestrictSimultaneous.contains(CardLoader.getCardByImageSet(imageSet)));
    }

    public boolean containsUnreleasedCards() {
        return this.getDeckList().stream().anyMatch(imageSet -> CardLoader.getCardByImageSet(imageSet).getReleaseDate() != null);
    }

    public boolean checkSimultaneousConflicts(Card cardObject) {
        return cardObject.getBanlistSimultaneousNum() > 0 && this.getDeckList().stream().anyMatch(imageSet -> cardObject.getBanlistSimultaneousList().contains(CardLoader.getCardByImageSet(imageSet)));
    }

    public boolean isLRIGLimitMismatch(Card cardObject) {
        if (CardConst.CardType.isLRIG(cardObject.getType()) || cardObject.getLRIGType().getPrimaryValue() == CardConst.CardLRIGType.NO_LRIG_LIMIT) {
            return false;
        }
        HashSet<CardConst.CardLRIGType> listUniqueLRIGTypes = new HashSet<CardConst.CardLRIGType>();
        for (String imageSet : this.deckLRIG) {
            Card cardLRIGObject = CardLoader.getCardByImageSet(imageSet);
            if (cardLRIGObject.getType() != CardConst.CardType.LRIG) continue;
            for (CardConst.CardLRIGType cardLRIGType2 : CardConst.CardLRIGType.values()) {
                if (!cardLRIGObject.getLRIGType().matches(cardLRIGType2)) continue;
                listUniqueLRIGTypes.add(cardLRIGType2);
            }
        }
        return listUniqueLRIGTypes.stream().noneMatch(cardLRIGType -> cardObject.getLRIGType().matches((CardConst.CardLRIGType)((Object)cardLRIGType)));
    }

    public boolean isPlayFormatCompatible(CardConst.PlayFormat playFormat) {
        return this.getDeckList().stream().allMatch(imageSet -> CardLoader.getCardByImageSet(imageSet).isAllowedInPlayFormat(playFormat));
    }

    public boolean isNotDoubleFacedCardPair(String imageSet) {
        return CardLoader.getCardByImageSet(imageSet).getDoubleFacedCardPairObject() == null || CardLoader.getCardByImageSet(imageSet).getCardFace() == CardConst.CardFace.FRONT;
    }

    public int getLRIGCardsLimit() {
        return this.getZeroLevelLRIGCount() < 3 ? 10 : 12;
    }

    public int getPieceCount() {
        return (int)this.deckLRIG.stream().filter(imageSet -> CardLoader.getCardByImageSet(imageSet).getType() == CardConst.CardType.PIECE).count();
    }

    public DeckStatus getValidStatus() {
        int countZeroLevelLRIG = this.getZeroLevelLRIGCount();
        if (countZeroLevelLRIG == 0) {
            return DeckStatus.ZEROLRIG;
        }
        int countPiece = this.getPieceCount();
        if (countPiece > 0 && countZeroLevelLRIG != 3) {
            return DeckStatus.CARDLIMIT_PIECE_LRIG;
        }
        if (this.getTotalCards(DeckType.MAIN) != 40) {
            return DeckStatus.CARDLIMIT_MAIN;
        }
        int countLRIGDeck = this.getTotalCards(DeckType.LRIG);
        if (countLRIGDeck > this.getLRIGCardsLimit()) {
            return DeckStatus.CARDLIMIT_LRIG;
        }
        if (this.getLifeBurstCount() > 20) {
            return DeckStatus.CARDLIMIT_LIFEBURST;
        }
        if (countPiece > 2 || countLRIGDeck - countPiece > 10) {
            return DeckStatus.CARDLIMIT_PIECE;
        }
        if (this.getPieceTeamLimitCount() > 1) {
            return DeckStatus.CARDLIMIT_PIECE_TEAM;
        }
        if (this.containsRestrictedCards()) {
            return DeckStatus.RESTRICTED;
        }
        if (this.containsUnreleasedCards()) {
            return DeckStatus.UNRELEASED;
        }
        return DeckStatus.READY;
    }

    public boolean containsFormatRestrictedCards(CardConst.PlayFormat playFormat) {
        if (playFormat != CardConst.PlayFormat.DIVA) {
            return false;
        }
        if (!this.checkDivaMatchingLRIGColors().isEmpty()) {
            return true;
        }
        List<String> deckList = this.getDeckList();
        for (String imageSet : deckList) {
            Card cardObject = CardLoader.getCardByImageSet(imageSet);
            if (cardObject.getBanlistRestriction() != RestrictedCardsParser.RestrictType.UP_TO_TWO || this.getCardCount(cardObject) <= 2) continue;
            return true;
        }
        return false;
    }

    public String exportAsCompressedData() {
        LinkedHashMap<String, Object> cachePrefix = new LinkedHashMap<String, Object>();
        HashMap<String, Integer> cacheCount = new HashMap<String, Integer>();
        for (String imageSet : this.getDeckList()) {
            int count;
            int tokenPrefix = imageSet.indexOf(45);
            if (tokenPrefix == -1) {
                System.err.println("Error: ImageSet \"" + imageSet + "\" lacks prefix! Unable to export invalid deck data.");
                return "";
            }
            String imageSetPrefix = imageSet.substring(0, tokenPrefix);
            String imageSetNumber = imageSet.substring(tokenPrefix + 1);
            if (!cacheCount.containsKey(imageSet)) {
                count = 1;
                cacheCount.put(imageSet, 1);
            } else {
                count = (Integer)cacheCount.get(imageSet) + 1;
                cacheCount.replace(imageSet, count);
            }
            if (!cachePrefix.containsKey(imageSetPrefix)) {
                cachePrefix.put(imageSetPrefix, imageSetNumber + ",");
                continue;
            }
            Object value = (String)cachePrefix.get(imageSetPrefix);
            switch (count) {
                case 1: {
                    value = (String)value + imageSetNumber + ",";
                    break;
                }
                case 2: {
                    value = ((String)value).replace(imageSetNumber, imageSetNumber + ":2");
                    break;
                }
                default: {
                    value = ((String)value).replace(imageSetNumber + ":" + (count - 1), imageSetNumber + ":" + count);
                }
            }
            cachePrefix.replace(imageSetPrefix, value);
        }
        StringBuilder data = new StringBuilder();
        for (String imageSetPrefix : cachePrefix.keySet()) {
            data.append(imageSetPrefix).append("=").append((CharSequence)cachePrefix.get(imageSetPrefix), 0, ((String)cachePrefix.get(imageSetPrefix)).length() - 1).append("|");
        }
        return !data.isEmpty() ? CardDataImageSet.compressImageSetData(data.substring(0, data.length() - 1)) : "";
    }

    public static boolean isMainDeckCard(Card cardObject) {
        return !(cardObject.getType() != CardConst.CardType.SIGNI && cardObject.getType() != CardConst.CardType.SPELL || cardObject.getDoubleFacedCardPairObject() != null && cardObject.getDoubleFacedCardPairObject().getType() == CardConst.CardType.PIECE);
    }

    public static enum DeckType {
        MAIN,
        LRIG;

    }

    public static enum DeckStatus {
        READY,
        ZEROLRIG,
        CARDLIMIT_LIFEBURST,
        CARDLIMIT_MAIN,
        CARDLIMIT_LRIG,
        CARDLIMIT_PIECE_LRIG,
        CARDLIMIT_PIECE_TEAM,
        CARDLIMIT_PIECE,
        RESTRICTED,
        RESTRICTED_SPECIAL,
        UNRELEASED;

    }

    public static enum DeckPosition {
        TOP,
        BOTTOM;

    }
}

